/*
 * Decompiled with CFR 0.152.
 */
package ru.turikhay.tlauncher.managers;

import com.google.gson.JsonParser;
import com.google.gson.JsonSyntaxException;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.IOException;
import java.io.Reader;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.io.IOUtils;
import org.apache.http.client.fluent.Request;
import org.apache.http.client.fluent.Response;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.turikhay.tlauncher.TLauncher;
import ru.turikhay.tlauncher.repository.Repository;
import ru.turikhay.tlauncher.ui.ConnectivityWarning;
import ru.turikhay.tlauncher.ui.notification.Notification;
import ru.turikhay.util.Lazy;
import ru.turikhay.util.SwingUtil;
import ru.turikhay.util.U;
import ru.turikhay.util.async.AsyncThread;

public class ConnectivityManager {
    private static final Logger LOGGER = LogManager.getLogger(ConnectivityManager.class);
    private static final String NOTIFICATION_ID = "connectivity";
    private final TLauncher launcher;
    private final List<Entry> entries;
    private final AtomicBoolean showFailedNotification = new AtomicBoolean();
    private ConnectivityWarning warningWindow;

    public ConnectivityManager(TLauncher launcher, List<Entry> entries) {
        this.launcher = launcher;
        this.entries = entries;
    }

    public void queueChecks() {
        this.entries.stream().filter(Entry::isNormalPriority).forEach(this::queueCheck);
    }

    public void queueCheck(Entry entry) {
        CompletableFuture<Boolean> future = entry.checkConnection();
        if (entry.isNormalPriority()) {
            future.whenCompleteAsync((r, e) -> this.notifyIfFailed(r != Boolean.TRUE || e != null));
        } else {
            future.whenCompleteAsync((r, e) -> this.scheduleUpdateWarningWindow());
        }
    }

    private void queueLowPriorityChecks() {
        this.entries.stream().filter(Entry::isLowPriority).forEach(this::queueCheck);
    }

    public void showNotificationOnceIfNeeded() {
        if (this.entries.stream().filter(Entry::isDone).allMatch(e -> e.isReachable() || e.isLowPriority())) {
            return;
        }
        if (!this.showFailedNotification.compareAndSet(false, true)) {
            return;
        }
        this.launcher.executeWhenReady(() -> SwingUtil.later(() -> this.launcher.getFrame().mp.defaultScene.notificationPanel.addNotification(NOTIFICATION_ID, new Notification("warning", this::showWarningWindow))));
    }

    private void notifyIfFailed(boolean failed) {
        if (failed) {
            this.queueLowPriorityChecks();
            this.showNotificationOnceIfNeeded();
        }
        this.scheduleUpdateWarningWindow();
    }

    private void scheduleUpdateWarningWindow() {
        SwingUtil.later(this::updateWarningWindow);
    }

    private void showWarningWindow() {
        if (this.warningWindow == null) {
            this.warningWindow = new ConnectivityWarning();
            this.warningWindow.addWindowListener(new WindowAdapter(){

                @Override
                public void windowClosed(WindowEvent e) {
                    ConnectivityManager.this.warningWindow = null;
                }
            });
            this.updateWarningWindow();
            this.warningWindow.pack();
            this.warningWindow.showAtCenter();
        } else {
            this.warningWindow.requestFocus();
        }
        this.launcher.getFrame().mp.defaultScene.notificationPanel.removeNotification(NOTIFICATION_ID);
    }

    private void updateWarningWindow() {
        if (this.warningWindow != null) {
            this.warningWindow.updateEntries(Collections.unmodifiableList(this.entries));
        }
    }

    public static Entry checkByContent(String name, String url, String expectedContent) {
        return new Entry(name, U.parseHost(url), (EntryChecker)new HttpContentChecker(url, expectedContent));
    }

    public static Entry checkByValidJson(String name, String url) {
        return new Entry(name, U.parseHost(url), (EntryChecker)new JsonContentChecker(url));
    }

    public static Entry checkRepoByValidJson(String name, Repository repository, String path) {
        return new Entry(name, repository.getList().getRelevant().getList().stream().flatMap(l -> l.getHosts().stream()), (EntryChecker)new RepoEntryJsonChecker(repository, path));
    }

    public static Entry forceFailed(String name) {
        return new Entry(name, Collections.emptyList(), () -> Boolean.FALSE);
    }

    public static class Entry {
        private final String name;
        private final Set<String> hosts;
        private final EntryChecker checker;
        private final Lazy<CompletableFuture<Boolean>> future;
        private final Lazy<CompletableFuture<Boolean>> completion;
        private int priority;

        private Entry(String name, Collection<String> hosts, EntryChecker checker) {
            this.name = name;
            this.hosts = Collections.unmodifiableSet(new LinkedHashSet<String>(hosts));
            this.checker = checker;
            this.completion = Lazy.of(CompletableFuture::new);
            this.future = Lazy.of(() -> {
                CompletableFuture<Boolean> f = CompletableFuture.supplyAsync(() -> {
                    try {
                        return checker.checkConnection();
                    }
                    catch (RuntimeException e) {
                        throw e;
                    }
                    catch (Exception e) {
                        throw new RuntimeException(e);
                    }
                }, AsyncThread.SHARED_SERVICE);
                f.whenCompleteAsync((r, e) -> {
                    CompletableFuture<Boolean> cf = this.completion.get();
                    if (e != null) {
                        cf.complete(Boolean.FALSE);
                    } else {
                        cf.complete((Boolean)r);
                    }
                }, (Executor)AsyncThread.SHARED_SERVICE);
                return f;
            });
        }

        private Entry(String name, Stream<String> hostStream, EntryChecker checker) {
            this(name, hostStream.collect(Collectors.toList()), checker);
        }

        Entry(String name, String host, EntryChecker checker) {
            this(name, Collections.singletonList(host), checker);
        }

        public String getName() {
            return this.name;
        }

        public Set<String> getHosts() {
            return this.hosts;
        }

        public EntryChecker getChecker() {
            return this.checker;
        }

        public boolean isQueued() {
            return this.future.isInitialized();
        }

        public boolean isDone() {
            return this.future.valueIfInitialized().map(CompletableFuture::isDone).orElse(false);
        }

        public boolean isReachable() {
            return this.future.valueIfInitialized().filter(f -> !f.isCompletedExceptionally()).map(f -> f.getNow(Boolean.FALSE)).orElse(Boolean.FALSE);
        }

        public int getPriority() {
            return this.priority;
        }

        boolean isNormalPriority() {
            return !this.isLowPriority();
        }

        boolean isLowPriority() {
            return this.priority < 0;
        }

        public Entry withPriority(int priority) {
            this.priority = priority;
            return this;
        }

        public CompletableFuture<Boolean> getTask() {
            return this.completion.get();
        }

        protected CompletableFuture<Boolean> checkConnection() {
            return this.future.get();
        }
    }

    private static class HttpContentChecker
    extends AbstractHttpGetEntryChecker {
        private final String expectedContent;

        protected HttpContentChecker(String url, String expectedContent) {
            super(url);
            this.expectedContent = Objects.requireNonNull(expectedContent, "expectedContent");
        }

        @Override
        protected void checkResponse(Response response) throws IOException {
            String actualContent = response.returnContent().asString();
            if (!this.expectedContent.equals(actualContent)) {
                throw new ContentMismatchException(this.expectedContent, actualContent);
            }
        }

        private static class ContentMismatchException
        extends IOException {
            public ContentMismatchException(String expected, String actual) {
                super(String.format(Locale.ROOT, "expected: \"%s\", actual: \"%s\"", expected, actual));
            }
        }
    }

    public static interface EntryChecker {
        public Boolean checkConnection() throws Exception;
    }

    private static class JsonContentChecker
    extends AbstractHttpGetEntryChecker {
        private JsonContentChecker(String url) {
            super(url);
        }

        @Override
        protected void checkResponse(Response response) throws IOException {
            String content = response.returnContent().asString();
            try {
                JsonParser.parseString((String)content);
            }
            catch (JsonSyntaxException e) {
                throw new InvalidJsonException(content, e);
            }
        }
    }

    private static class RepoEntryJsonChecker
    implements EntryChecker {
        private final Repository repo;
        private final String path;

        public RepoEntryJsonChecker(Repository repo, String path) {
            this.repo = repo;
            this.path = path;
        }

        @Override
        public Boolean checkConnection() throws IOException {
            try {
                String content = IOUtils.toString((Reader)this.repo.read(this.path));
                try {
                    JsonParser.parseString((String)content);
                }
                catch (JsonSyntaxException e) {
                    throw new InvalidJsonException(content, e);
                }
            }
            catch (IOException e) {
                LOGGER.warn("Connectivity check to {} (using {}) failed", (Object)this.path, (Object)this.repo.name(), (Object)e);
                throw e;
            }
            return Boolean.TRUE;
        }
    }

    private static class InvalidJsonException
    extends IOException {
        public InvalidJsonException(String content, Throwable cause) {
            super("invalid json: " + content, cause);
        }
    }

    private static abstract class AbstractHttpGetEntryChecker
    implements EntryChecker {
        private final String url;

        protected AbstractHttpGetEntryChecker(String url) {
            this.url = url;
        }

        @Override
        public final Boolean checkConnection() {
            try {
                this.checkResponse(Request.Get((String)this.url).execute());
            }
            catch (IOException e) {
                LOGGER.warn("Connectivity check to {} failed", (Object)this.url, (Object)e);
                throw new RuntimeException(e);
            }
            return Boolean.TRUE;
        }

        protected abstract void checkResponse(Response var1) throws IOException;
    }
}

